<?php
// Bootstrap: load config, ensure DB, define helpers

$config = require __DIR__ . '/config.php';

if (empty($config['bot_token'])) {
    http_response_code(500);
    echo "BOT_TOKEN missing. Configure .env";
    exit;
}

// Ensure storage paths exist
@mkdir(__DIR__ . '/../storage/database', 0775, true);
@mkdir(__DIR__ . '/../storage/logs', 0775, true);

// SQLite PDO
try {
    $pdo = new PDO('sqlite:' . $config['db']['path']);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->exec("CREATE TABLE IF NOT EXISTS orders (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        chat_id TEXT NOT NULL,
        amount INTEGER NOT NULL,
        order_id TEXT NOT NULL UNIQUE,
        payment_url TEXT,
        status TEXT NOT NULL DEFAULT 'PENDING',
        currency TEXT NOT NULL DEFAULT 'INR',
        customer_mobile TEXT DEFAULT NULL,
        created_at INTEGER NOT NULL,
        updated_at INTEGER NOT NULL
    );");
} catch (Throwable $e) {
    file_put_contents(__DIR__ . '/../storage/logs/error.log', date('c') . ' DB error: ' . $e->getMessage() . PHP_EOL, FILE_APPEND);
    http_response_code(500);
    echo "DB error. Check storage/logs/error.log";
    exit;
}

// Helpers
function tg_api($method, $params = []) {
    global $config;
    $url = "https://api.telegram.org/bot{$config['bot_token']}/" . $method;
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
        CURLOPT_POSTFIELDS => json_encode($params),
    ]);
    $resp = curl_exec($ch);
    if (curl_errno($ch)) {
        file_put_contents(__DIR__ . '/../storage/logs/error.log', date('c') . ' TG error: ' . curl_error($ch) . PHP_EOL, FILE_APPEND);
        curl_close($ch);
        return false;
    }
    curl_close($ch);
    return json_decode($resp, true);
}

function respond_json($data, $code = 200) {
    http_response_code($code);
    header('Content-Type: application/json');
    echo json_encode($data);
    exit;
}

function payin_xverify(array $data, $secret) {
    ksort($data);
    $pairs = [];
    foreach ($data as $k => $v) {
        $pairs[] = $k . '=' . $v;
    }
    $dataString = implode('|', $pairs);
    return hash_hmac('sha256', $dataString, $secret);
}

function http_post_json($url, $data, $headers = []) {
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_HTTPHEADER => array_merge(['Content-Type: application/json'], $headers),
        CURLOPT_POSTFIELDS => json_encode($data),
        CURLOPT_SSL_VERIFYPEER => true,
        CURLOPT_SSL_VERIFYHOST => 2,
        CURLOPT_TIMEOUT => 30,
    ]);
    $resp = curl_exec($ch);
    if (curl_errno($ch)) {
        $err = curl_error($ch);
        curl_close($ch);
        return [false, $err];
    }
    $status = curl_getinfo($ch, CURLINFO_RESPONSE_CODE);
    curl_close($ch);
    return [$status, $resp];
}

function now() { return time(); }
