<?php
// helpers.php - utility functions, LG Pay helpers, UI helpers
require_once __DIR__ . '/db.php';
require_once __DIR__ . '/config.php';

function tgRequest($method, $params=[]){
  $url = "https://api.telegram.org/bot" . BOT_TOKEN . "/" . $method;
  $ch = curl_init($url);
  curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
  curl_setopt($ch, CURLOPT_POST, true);
  curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($params));
  curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
  curl_setopt($ch, CURLOPT_TIMEOUT, 10);
  $res = curl_exec($ch);
  if(curl_errno($ch)){
    // optionally log curl error
  }
  curl_close($ch);
  return json_decode($res, true);
}

function sendMessage($chat_id, $text, $extra=[]){
  $params = array_merge(['chat_id'=>$chat_id,'text'=>$text,'parse_mode'=>'HTML','disable_web_page_preview'=>false], $extra);
  return tgRequest('sendMessage', $params);
}

function sendPhoto($chat_id, $photo_url, $caption = '', $extra = []){
  $params = array_merge([
    'chat_id' => $chat_id,
    'photo' => $photo_url,
    'caption' => $caption,
    'parse_mode' => 'HTML'
  ], $extra);
  return tgRequest('sendPhoto', $params);
}

function sendInlineKeyboard($chat_id, $text, $inline_keyboard){
  $reply_markup = ['inline_keyboard'=>$inline_keyboard];
  return sendMessage($chat_id, $text, ['reply_markup'=>json_encode($reply_markup)]);
}

// Fetch active giveaways
function listActiveGiveaways(){
  $pdo = getPDO();
  $stmt = $pdo->query("SELECT * FROM giveaways WHERE is_active=1 ORDER BY created_at DESC");
  return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// Progress bar helper
function generateProgressBar($current, $total, $length = 12){
  if($total <= 0) $total = 1;
  $ratio = max(0, min(1, $current / $total));
  $filled = (int) round($ratio * $length);
  $empty = $length - $filled;
  $filledChar = '█';
  $emptyChar = '░';
  $bar = str_repeat($filledChar, $filled) . str_repeat($emptyChar, $empty);
  $percent = (int) round($ratio * 100);
  return ['bar' => $bar, 'percent' => $percent];
}

// Build keyboard for giveaway card
function buildGiveawayKeyboard($giveaway){
  $joined = intval($giveaway['joined_count']);
  $slots = intval($giveaway['total_slots'] ?? 1);
  $price = number_format($giveaway['price'],2);
  $join_text = ($joined >= $slots) ? "Sold Out" : "Join — ₹{$price}";
  $join_cb = ($joined >= $slots) ? "soldout:{$giveaway['id']}" : "join:{$giveaway['id']}";
  $keyboard = [
    [
      ['text'=>$join_text, 'callback_data'=>$join_cb],
      ['text'=>'Details', 'callback_data'=>'details:'.$giveaway['id']]
    ],
    [
      ['text'=>'Share', 'callback_data'=>'share:'.$giveaway['id']]
    ]
  ];
  return $keyboard;
}

// Render giveaway caption with progress bar
function giveawayCaption($g){
  $joined = intval($g['joined_count']);
  $slots = intval($g['total_slots'] ?? 1);
  $remaining = max(0, $slots - $joined);
  $p = generateProgressBar($joined, $slots, 12);
  $bar = $p['bar'];
  $percent = $p['percent'];

  $caption = "<b>".htmlspecialchars($g['title'])."</b>\n\n";
  if(!empty($g['prize'])) $caption .= "🎁 Prize: <b>".htmlspecialchars($g['prize'])."</b>\n";
  $caption .= "💵 Entry Fee: <b>₹".number_format($g['price'],2)."</b>\n";
  $caption .= "👥 Joined: <b>{$joined}</b> / <b>{$slots}</b>\n";
  $caption .= "⏳ Slots left: <b>{$remaining}</b>\n\n";
  $caption .= "<code>{$bar}</code> <b>{$percent}%</b>\n\n";
  if(!empty($g['description'])) $caption .= "<i>".htmlspecialchars(substr($g['description'],0,350)).(strlen($g['description'])>350?'...':'')."</i>\n";
  return $caption;
}

// ---------------- LG Pay helpers ----------------
function lgpaySign($params){
  ksort($params);
  $str = '';
  foreach($params as $k=>$v){
    if($v === '' || $v === null) continue;
    if($k === 'sign') continue;
    $str .= "$k=$v&";
  }
  $str .= 'key='.LG_PAY_SECRET;
  return strtoupper(md5($str));
}

function lgpayCreateOrder($params){
  $url = 'https://www.lg-pay.com/api/order/create';
  $ch = curl_init($url);
  curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
  curl_setopt($ch, CURLOPT_POST, true);
  curl_setopt($ch, CURLOPT_POSTFIELDS, $params);
  curl_setopt($ch, CURLOPT_TIMEOUT, 15);
  $res = curl_exec($ch);
  curl_close($ch);
  $json = json_decode($res, true);
  return $json ?: ['status'=>0,'raw'=>$res];
}

// ---------------- Payment flow ----------------
function startPayment($user_id, $giveaway_id, $amount){
  $pdo = getPDO();

  // prevent duplicate pending/paid entries
  $stmt = $pdo->prepare("SELECT id FROM orders WHERE user_id=? AND giveaway_id=? AND status IN ('pending','paid') LIMIT 1");
  $stmt->execute([$user_id,$giveaway_id]);
  if($stmt->fetch()){
    return ['error' => 'You already have a pending/paid entry for this giveaway.'];
  }

  $stmt = $pdo->prepare("INSERT INTO orders (user_id,giveaway_id,amount,status) VALUES (?, ?, ?, 'pending')");
  $stmt->execute([$user_id,$giveaway_id,$amount]);
  $order_id = $pdo->lastInsertId();

  $order_sn = uniqid('pay_');
  $money = intval($amount * 100);

  $params = [
    'app_id' => LG_PAY_APPID,
    'trade_type' => 'INRUPI',
    'order_sn' => $order_sn,
    'money' => $money,
    'notify_url' => LG_PAY_NOTIFY_URL,
    'ip' => $_SERVER['REMOTE_ADDR'] ?? '127.0.0.1',
    'remark' => json_encode(['user_id'=>$user_id,'order_id'=>$order_id,'giveaway_id'=>$giveaway_id])
  ];
  $params['sign'] = lgpaySign($params);
  $resp = lgpayCreateOrder($params);

  $stmt = $pdo->prepare("UPDATE orders SET payment_payload = ? WHERE id = ?");
  $stmt->execute([json_encode(['order_sn'=>$order_sn,'raw_resp'=>$resp]), $order_id]);

  if(isset($resp['status']) && $resp['status'] == 1 && isset($resp['data']['pay_url'])){
    return ['order_id'=>$order_id,'payment_url'=>$resp['data']['pay_url'],'order_sn'=>$order_sn];
  }
  return ['order_id'=>$order_id,'payment_url'=>null,'error'=>'Failed to create payment link'];
}

function verifyPaymentAndMarkPaid($order_id, $payment_data){
  $pdo = getPDO();
  $stmt = $pdo->prepare("UPDATE orders SET status='paid', payment_payload = ? WHERE id = ?");
  $stmt->execute([json_encode($payment_data), $order_id]);

  $stmt = $pdo->prepare("UPDATE giveaways SET joined_count = joined_count + 1 WHERE id = (SELECT giveaway_id FROM orders WHERE id=?)");
  $stmt->execute([$order_id]);
}

// simple notify with gif
function notifyWithGif($chat_id, $text, $gif_url){
  tgRequest('sendAnimation', ['chat_id'=>$chat_id,'animation'=>$gif_url,'caption'=>$text,'parse_mode'=>'HTML']);
}
?>